/*
 * Activity.java
 *
 * Created 12.10.2000
 * Documented 7.7.2001
 */

package timetable.data;

import java.util.Vector;
import timetable.util.*;

/** Reprezentece jedn aktivity v rozvrhovacm problmu. 
 * <BR>M tak metody pro pidn a odebrn aktivity z rozvrhu, kontrolu splnn zvislost, zjiovn potu a seznamu monch umstn v rozvrhu i vpoet potu kolidujcch aktivit pi zvolen danho umstn.
 * @author Tomas Muller
 * @version 1.0.1
 */
public class Activity extends java.lang.Object {
    /** Maximln dlka zkratky jmna.*/
    public static int SHORT_CUT_MAX_LENGTH = 7;
    /** Globln konfigurace rozvrhovae. */
    protected Config config = null;
    
    /** Nkolika psmen zkratka jmna aktivity, pro zobrazen. */
    public String shortCut = null;
    /** Jmno aktivity. */
    public String name = null;    
    /** Dlka trvn aktivity. */
    public int length = 1;
    /** Poznmka. */
    public String note = null;
    /** asov preference aktivity - soft a hard podmnky na jednotliv asov sloty. */
    protected TimePreference preference = null;
    /** Zdroje, kter aktivita me vyuvat. Jde o mnoinu skupin zdroj. */
    public ActivityResources resources = new ActivityResources();
    
    /** Naplnovan zatek aktivity. Pokud je aktivita nenaplnovan je hodnota tto promnn -1.*/
    public int start = -1;
    /** Zdroje, kter aktivita vyuv, je-li naplnovan. Promnn mus mt stejnou strukturu, jako promnn resources.
     * V ppad skupiny alternativnch zdroj, zde tato skupina obsahuje pouze jeden zdroj.
     */
    public ActivityResources selectedResources = null;
    
    /** Vstupn rozvrh: zatek aktivity*/
    public int lastStart = -1;
    /** Vstupn rozvrh: poadovan zdroje*/
    public ActivityResources lastSelectedResources = null;
    /** Nejlep nalezen rozvrh: zatek aktivity*/
    public int bestStart = -1;
    /** Nejlep nalezen rozvrh: poadovan zdroje*/
    public ActivityResources bestSelectedResources = null;

    /** Pro plnovn, eviduje kter aktivita tuto aktivitu naposledy vyhodila z rozvrhu.*/
    public Activity lastRemovedBy = null;
    /** Lze ji naplnovanou aktivitu peplnovat ? (Tedy odebrat z rozvrhu a zaadit na jin msto.)*/
    public boolean canReschedule = true;
    /** Kolikrt byla aktivita pi plnovn vyhozena z rozvrhu. Promnn pomh urit nejvce vadc (nejhe naplnovateln) aktivity. Pi sputn eie se vdy vynuluje.*/
    public int nrRemoves=0;
    
    /** Konstruktor.
     * @param config konfigurace
     * @param shortCut uniktn zkrcen jmnno aktivity - pro vizualizaci rozvrhu
     * @param name jmno aktivity
     * @param note poznmka
     * @param length dlka trvn aktivity
     */    
    public Activity(Config config,String shortCut,String name,String note,int length) throws TimetableException {
        this.config = config;
        this.name=name;
        this.shortCut=(shortCut==null||shortCut.length()<=SHORT_CUT_MAX_LENGTH?shortCut:shortCut.substring(0,SHORT_CUT_MAX_LENGTH));
        this.note=note;
        this.length=length;
        preference = new TimePreference(config);
    }
    
    /** Konstruktor.
     * @param config konfigurace
     * @param name jmno aktivity
     * @param length dlka trvn aktivity
     */    
    public Activity(Config config,String name,int length) throws TimetableException {
        this(config, name, name,null, length);
    }

    /** Konstruktor.
     * @param config konfigurace
     */    
    public Activity(Config config) throws TimetableException {
        this(config, null,null, null, 1);
    }
    
    /** Inicializace pomocnch promnnch ped dalm plnovnm. Automaticky volno rozvrhovaem. 
     * Jde o vynulovn potu vyhozen a aktivyty, kter naposled tuto aktivitu vyhodila z rozvrhu 
     */    
    public void reset() {
        nrRemoves=0;
        lastRemovedBy=null;
    }
        
    /** Zjiuje, zda-li je aktivita ji naplnovan. 
     * @return <CODE>true</CODE> pokud je aktivita naplnovan
     */    
    public boolean isScheduled() {
        return (start>=0);
    }
    
    /** Vrt tabulku asovch preferenc
     * @return tabulka asovch preferenc
     */
    public TimePreference getPreference() {
        return preference;
    }
    
    /** Nastav tabulku asovch preferenc
     * @param preference tabulka asovch preferenc
     */
    public void setPreference(TimePreference preference) {
        this.preference=preference;
    }

    /** Odebrn aktivity z rozvrhu. Tedy nastaven zatku na -1 a jej odebrn z asovch tabulek zdroj 
     *  @param removedBy aktivita, kvli kter se tato aktivita odebr
     */    
    public void remove(Activity removedBy) {
        if (start==-1) return;
        for (int i=0;i<selectedResources.size();i++) 
              if (selectedResources.isGroup(i)) {
                  for (int k=0;k<selectedResources.getResources(i).size();k++)
                      for (int j=0;j<length;j++) 
                        selectedResources.getResources(i).get(k).setActivityAt(j+start,null);
                } else 
                    for (int j=0;j<length;j++) 
                      selectedResources.getResource(i).setActivityAt(j+start,null);
        selectedResources.removeAll();
        start=-1;
        nrRemoves++;
        this.lastRemovedBy=removedBy;
    }
        
    /** Naplnovn aktivity. Tedy nastaven jejho zatku a propagace alokace jednotlivch asovch slot do asovch tabulek vybranch zdroj. 
     * <BR>Vechny konfliktn aktivity ji mus bt odebrny !!!
     * @param slot nov zatek aktivity - asov slot
     * @param selectedResources vybran zdroje
     */    
    public void schedule(int slot, ActivityResources selectedResorces) {
        start = slot;
        this.selectedResources = selectedResorces;
        if (start<0) return;
        for (int i=0;i<selectedResources.size();i++) {
            if (selectedResources.isGroup(i))
                for (int k=0;k<selectedResources.getResources(i).size();k++)
                    for (int j=0;j<length;j++) 
                        selectedResources.getResources(i).get(k).setActivityAt(j+start,this);
            else 
                for (int j=0;j<length;j++) 
                    selectedResources.getResource(i).setActivityAt(j+start,this);
        }
    }
    
    /** Zjiuje konzistenci naplnovn aktivity. Jmenovit: 
     * <ul><li>Naplnovn aktivity nepesahuje poet monch slot</li>
     * <li>Naplnovn aktivity nepesahuje dan den</li>
     * <li>Nen poruena HARD podmnka v asovch preferencch aktivity? </li>
     * <li>Jsou v podku asov zvislosti, ve kterch je obsaena dan aktivita? </li>
     * <li>Konzistenci vybranch zdroj s mnoinou skupin poadovanch zdroj a jejich sprvn naalokace</li>
     * <li>Neporuen HARD podmnky v asovch preferencch vybranch zdruj touto aktivitou </li></ul>
     * @param dependencies asov zvislosti mezi aktivitami, null pokud nemaj bt kontrolovny
     * @return <CODE>true</CODE> pokud je ve v podku
     */
    public boolean isConsistent(ActivityDependenceGroup dependences) throws TimetableException {
        if (isScheduled()) {
            int slot=this.start;
            if (slot<0 || slot+length>config.getInt(Config.NR_SLOTS)) return false;
            int nrSlotsPerDay = config.getInt(Config.NR_SLOTS)/config.getInt(Config.NR_DAYS);
            if (slot/nrSlotsPerDay != (slot+length-1)/nrSlotsPerDay) return false;
            for (int j=0;j<length;j++) if (preference.isHARD(j+slot)) return false;
            if (dependences!=null && !dependences.isOK(this,slot)) return false;
            for (int i=0;i<selectedResources.size();i++) {
                if (selectedResources.isGroup(i)) {
                    for (int j=0;j<selectedResources.getResources(i).size();j++)
                        if (!selectedResources.getResources(i).get(j).isConsistentFor(this)) return false;
                    if (!resources.getResources(i).haveSameContent(selectedResources.getResources(i))) return false;
                } else {
                    if (!selectedResources.getResource(i).isConsistentFor(this)) return false;
                    if (resources.isGroup(i)) {
                        if (resources.getResources(i).indexOf(selectedResources.getResource(i))<0) return false;
                    } else {
                        if (!resources.getResource(i).equals(selectedResources.getResource(i))) return false;
                    }
                }
            }
        }
        return true;
    }
    
    /** Zjiuje, zda-li lze aktivitu naplnovat od danho asovho slotu. Tedy zda-li existuj zdroje, kter nemaj HARD podmnku v danch slotech. A zdali neexistuje podmnka mezi aktivitami, kter by byla poruena a druh z aktivit by nela odebrat.
     * @param slot zatek aktivity, pro kter je daj zjiovn
     * @param dependencies asov zvislosti mezi aktivitami, null pokud nemaj bt uvaovny
     * @return <CODE>true</CODE> pokud lze aktivita v danm ase naplnovat.
     */    
    public boolean canScheduleAt(int slot, ActivityDependenceGroup dependences) throws TimetableException {
        if (slot<0 || slot+length>config.getInt(Config.NR_SLOTS)) return false;
        int nrSlotsPerDay = config.getInt(Config.NR_SLOTS)/config.getInt(Config.NR_DAYS);
        if (slot/nrSlotsPerDay != (slot+length-1)/nrSlotsPerDay) return false;
        for (int j=0;j<length;j++) if (preference.isHARD(j+slot)) return false;
        if (dependences!=null && !dependences.isFixedOK(this,slot)) return false;
        return resources.canAllBeFreeAt(slot,length);
    }
    
    /** Zjiuje, zda-li lze aktivitu naplnovat do danho asovho slotu bez kolize. 
     * @param slot zatek aktivity, pro kter je daj zjiovn
     * @param dependencies asov zvislosti mezi aktivitami, null pokud nemaj bt uvaovny
     * @return <CODE>true</CODE> pokud lze aktivita v danm ase naplnovat bez kolize s jinou aktivitou.
     */    
    public boolean canScheduleWithoutCollisionAt(int slot, ActivityDependenceGroup dependences) throws TimetableException {
        if (slot<0 || slot+length>config.getInt(Config.NR_SLOTS)) return false;
        int nrSlotsPerDay = config.getInt(Config.NR_SLOTS)/config.getInt(Config.NR_DAYS);
        if (slot/nrSlotsPerDay != (slot+length-1)/nrSlotsPerDay) return false;
        for (int j=0;j<length;j++) if (preference.isHARD(j+slot)) return false;
        if (dependences!=null && !dependences.isOK(this,slot)) return false;
        return resources.isAllFreeAt(slot,length);
    }
    
    /** Zjiuje, poet mst, kam lze aktivita v danm ase naplnovat bez kolize. 
     * @param slot zatek aktivity, pro kter je daj zjiovn
     * @param dependencies asov zvislosti mezi aktivitami, null pokud nemaj bt uvaovny
     * @return poet mst, kam lze aktivita v danm ase naplnovat bez kolize s jinou aktivitou.
     */    
    public int countScheduleWithoutCollisionAt(int slot, ActivityDependenceGroup dependences) throws TimetableException {
        if (slot<0 || slot+length>config.getInt(Config.NR_SLOTS)) return 0;
        int nrSlotsPerDay = config.getInt(Config.NR_SLOTS)/config.getInt(Config.NR_DAYS);
        if (slot/nrSlotsPerDay != (slot+length-1)/nrSlotsPerDay) return 0;
        for (int j=0;j<length;j++) if (preference.isHARD(j+slot)) return 0;
        if (dependences!=null && !dependences.isOK(this,slot)) return 0;
        return resources.countAllFreeAt(slot,length);
    }
    
    /** Zjiuje, poet mst, kam lze aktivita bez kolize naplnovat. (Projde vechny sloty)
     * @param dependencies asov zvislosti mezi aktivitami, null pokud nemaj bt uvaovny
     * @return poet mst, kam lze aktivita naplnovat bez kolize s jinou aktivitou.
     */    
    public int countScheduleWithoutCollision( ActivityDependenceGroup dependences) throws TimetableException {
        int sum=0;
        for (int i=0;i<config.getInt(Config.NR_SLOTS);i++) 
            sum+=countScheduleWithoutCollisionAt(i, dependences);
        return sum;
    }

   /** Zjiuje poet asovch slot, kam lze aktivita naplnovat (i kdy spsob kolizi).
     * @param dependencies asov zvislosti mezi aktivitami, null pokud nemaj bt uvaovny
     * @return poet slot, kam lze aktivita naplnovat
    */
    public int countSchedule(ActivityDependenceGroup dependences) throws TimetableException {
        int sum=0;
        for (int i=0;i<config.getInt(Config.NR_SLOTS);i++) 
            if (canScheduleAt(i,dependences)) sum++;
        return sum;
    }
    
    /** Zpis aktivity do objektovho streamu. Nutnm parametrem je mnoina vech zdroj, ukldan zdroje jsou toti kdovny jako poad v tto mnoin. Pedchoz umstn a nejlep nalezen umstn nen zapisovno.
     * @param out stream
     * @param alResources mnoina vech zdroj
     */
     protected void write(java.io.ObjectOutputStream out, ResourceGroup allResources) throws java.io.IOException {
         out.writeInt(length);
         out.writeInt(start);
         out.writeObject(name);
         out.writeObject(shortCut);
         out.writeObject(note);
         out.writeObject(preference);
         resources.write(out, allResources);
         if (selectedResources==null || start<0) out.writeBoolean(false); else {out.writeBoolean(true); selectedResources.write(out, allResources);};
         out.writeBoolean(canReschedule);
     }
     
     /** Naten aktivity z streamu objekt.
      * @param in stream
      * @param config globln konfigurace
      * @param allResources mnoina vech zdroj
      * @return naten aktivita
      */
     protected static Activity read(java.io.ObjectInputStream in, Config config, ResourceGroup allResources) throws java.io.IOException, java.lang.ClassNotFoundException, TimetableException {
         Activity a=new Activity(config);
         a.length=in.readInt();
         a.start=in.readInt();
         a.name=(String)in.readObject();
         a.shortCut=(String)in.readObject();
         a.note=(String)in.readObject();
         a.preference=(TimePreference)in.readObject();
         a.preference.config=config;
         a.resources=ActivityResources.read(in,allResources);
         if (in.readBoolean()) a.selectedResources=ActivityResources.read(in,allResources); else a.selectedResources=null;
         a.lastStart=-1;//in.readInt();
         a.lastSelectedResources=null;//(ResourceGroup)in.readObject();
         a.lastRemovedBy=null;//(Aktivita)in.readObject();
         a.canReschedule=in.readBoolean();
         a.nrRemoves=0;
         return a;
     }

    
}
