/*
 * Solver.java
 *
 * Created on 17. prosinec 2000, 14:47
 * Documented 18. 7. 2001
 */

package timetable.solver;

import timetable.data.*;
import timetable.util.*;
import timetable.solver.strategy.*;
import timetable.solver.strategy.price.*;
import timetable.solver.strategy.value.*;
import timetable.solver.strategy.variable.*;
/** ei problmu interaktivnho rozvrhovn.
 *
 * @author Tom Mller
 * @version 1.0
 */
public class Solver extends Object {
    /** konfigurace */
    Config config;
    /** Problm - rozvrhovac data */
    Problem problem;
    /** tabu-list - pro vbr umstn */
    Tabu tabu = null;
    /** heuristika vbru nenaplnovan aktivity*/
    VariableSelectionInterface variableSelection = null;
    /**  heuristika vbru umstn*/
    ValueSelectionInterface valueSelection = null;
    
    /** poet iterac */
    public int nrIters = 0;
    
    /** doba een problmu*/
    public long solveTime = 0;
    /** doba generovn monch umstn */
    public long generateTime = 0;
    /** doba vbru umstn */
    public long selectValueTime = 0;
    /** doba vbru aktivity */
    public long selectVariableTime = 0;
    /** maximln poet naplnovanch aktivit, kterho se podailo doshnout */
    public long maxScheduledActivities = 0;

    /** Konstruktor.
     * @param config konfigurace
     * @param problem problm rozvrhu
     */
    public Solver(Config config, Problem problem) throws TimetableException {
        this.problem=problem;
        this.config=config;
        nrIters=0;
        solveTime=generateTime=selectValueTime=selectVariableTime=0;
        maxScheduledActivities=0;
        tabu = new Tabu(config);
    }
    
    /** Vrt problm - data.
     * @return problm rozvrhu
     */
    public Problem getProblem() {return problem; }
    
    /** Reset eie i problmu - odebrn vech naplnovanch aktivit z rozvrhu.
     */
    public void reset() throws TimetableException {
        nrIters=0;
        solveTime=generateTime=selectValueTime=selectVariableTime=0;
        maxScheduledActivities=0;
        problem.reset();
        tabu = new Tabu(config);
    }
    
    /** Nastaven heuristiky vbru nenaplnovan aktivity
     * @param variableSelection heuristika vbru nenaplnovan aktivity
     */
    public void setVariableSelection(VariableSelectionInterface variableSelection) {
        this.variableSelection=variableSelection;
    }
    
    /** Nastaven heuristiky vbru umstn 
     * @param variableSelection heuristika vbru umstn nenaplnovan aktivity
     */
    public void setValueSelection(ValueSelectionInterface valueSelection) {
        this.valueSelection=valueSelection;
    }
    
    /** een problmu.
     */
    public void solve() throws TimetableException {
        solveTime=generateTime=selectValueTime=selectVariableTime=0;
        maxScheduledActivities=0;
        nrIters = 0;
        problem.saveBestState();
        
        int canNotSchedule =0;

        long solveStart= System.currentTimeMillis();
        
        ActivityGroup unscheduled = problem.unscheduledActivities();
        
        while (unscheduled.size()>0 && nrIters<config.getInt(Config.NR_TRIES)) {
            config.log("Try: "+nrIters+", Unscheduled: "+unscheduled.size());
            if (maxScheduledActivities<problem.activities.size()-unscheduled.size()) {
              maxScheduledActivities=problem.activities.size()-unscheduled.size();
              problem.saveBestState();
            }
            
            long selectVariableStart=System.currentTimeMillis();;
              Activity activity=variableSelection.select(unscheduled,problem.dependences);
            selectVariableTime+=(System.currentTimeMillis()-selectVariableStart);
            
            long generateStart=System.currentTimeMillis();
              int slot = schedule(activity);
            generateTime+=(System.currentTimeMillis()-generateStart);
            
            if (slot<0) {
              canNotSchedule++;            
            } else {
              config.log("Activity: "+activity.name+" -> "+slot+" (resources: "+activity.selectedResources.toString()+" )");
              if (config.existKey("DEBUG_LOG") && problem.resources!=null) {
                  try { problem.print(config.getLogWriter(),problem.resources[0],0,problem.resources.length); } catch (Exception e) {}
              };
            };
            
            unscheduled = problem.unscheduledActivities();
            nrIters++;
        }
        
        if (maxScheduledActivities<problem.activities.size()-unscheduled.size())
            maxScheduledActivities=problem.activities.size()-unscheduled.size();
        
        solveTime=System.currentTimeMillis()-solveStart;
        generateTime-=selectValueTime;
    }
   
    /** Rozvrhne danou aktivitu.
     * @param activity nenaplnovan aktivita
     * @return poten slot (-1 pokud se nepodailo aktivitu naplnovat)
     */
    public int schedule(Activity activity) throws TimetableException {
        valueSelection.reset(activity);
        
        config.log("SCHEDULE: "+activity.name);

        for (int slot=0;slot<config.getInt(Config.NR_SLOTS);slot++) 
            if (activity.canScheduleAt(slot,null)) {
                ActivityGroup conflictActivities = problem.dependences.getConflictActivities(activity,slot);
                boolean can = true; 
                for (int i=0;i<conflictActivities.size();i++)
                    if (!conflictActivities.get(i).canReschedule) can=false;
                config.log("  -> "+slot+" (conflict: "+conflictActivities.toString()+")");
                if (can) generate(activity, valueSelection, slot, new ActivityResources(), conflictActivities);
            }
        
        long selectValueStart=System.currentTimeMillis();
        valueSelection.select();
        int selectedSlot=valueSelection.selectedSlot();
        selectValueTime+=(System.currentTimeMillis()-selectValueStart);
        
        if (selectedSlot<0) return selectedSlot;
        
        ActivityGroup conflictActivities = valueSelection.selectedConflictActivities();
        for (int i=0;i<conflictActivities.size();i++) 
            conflictActivities.get(i).remove(activity);
        activity.schedule(selectedSlot, valueSelection.selectedResources());
                
        return selectedSlot;
    }
    
    /** Rozvrhne danou aktivitu na dan msto (ureno pouze potenm slotem a jednm zdrojem - takovch mst me bt vce).
     * @param activity nenaplnovan aktivita
     * @param slot poten slot
     * @param useResource zdroj, kter mus bt pouit (null, pokud nen zvolen)
     * @return true, pokud se podailo aktivitu naplnovat
     */
    public boolean schedule(Activity activity, int slot, Resource useResource) throws TimetableException {
        valueSelection.reset(activity);
        
        config.log("SCHEDULE: "+activity.name);

        if (activity.canScheduleAt(slot,null)) {
            ActivityGroup conflictActivities = problem.dependences.getConflictActivities(activity,slot);
            //System.out.println("Activity:"+activity.shortCut+"\nSlot:"+slot+"\nConflicts:"+conflictActivities.toString());
            boolean can = true; 
            for (int i=0;i<conflictActivities.size();i++)
                if (!conflictActivities.get(i).canReschedule) can=false;
            config.log("  -> "+slot+" (conflict: "+conflictActivities.toString()+")");
            if (can) generate(activity, valueSelection, slot, new ActivityResources(), conflictActivities,useResource);
        }
        
        long selectValueStart=System.currentTimeMillis();
        valueSelection.select();
        int selectedSlot=valueSelection.selectedSlot();
        selectValueTime+=(System.currentTimeMillis()-selectValueStart);
        
        if (selectedSlot<0) return false;
        
        ActivityGroup conflictActivities = valueSelection.selectedConflictActivities();
        for (int i=0;i<conflictActivities.size();i++) 
            conflictActivities.get(i).remove(activity);
        activity.schedule(selectedSlot, valueSelection.selectedResources());
                
        return true;
    }
    
    /** Rekuzivn metoda, kter generuje mon umstn aktivity a pidv je do heuristiky vbru umstn.
     * @param activity nenaplnovan aktivita, pro kterou se umstn hled
     * @param valueSelection heuristika vbru umstn
     * @param slot poten slot, pro kter se umstn generuj
     * @param resources ji vybran zdroje (pro rekurzi, volno s new ActivityResources())
     * @param conflict mnoina ji nalezench konfliktnch aktivit
     */    
    private void generate(Activity activity, ValueSelectionInterface valueSelection, int slot, ActivityResources resources, ActivityGroup conflict) throws TimetableException {
        int number= resources.size();
        if (number<activity.resources.size()) {
            if (activity.resources.isGroup(number)) {
                if (activity.resources.getResources(number).conjunctive) {
                    resources.add(activity.resources.getResources(number));
                    generate(activity,valueSelection,slot,resources,conflict);
                    resources.remove(activity.resources.getResources(number));
                } else {
                    for (int i=0;i<activity.resources.getResources(number).size();i++) 
                        if (activity.resources.getResources(number).get(i).canBeFreeAt(slot,activity.length)) {
                            resources.add(activity.resources.getResources(number).get(i));
                            generate(activity,valueSelection,slot,resources,conflict);
                            resources.remove(activity.resources.getResources(number).get(i));
                        }
                }
            } else {
                resources.add(activity.resources.getResource(number));
                generate(activity,valueSelection,slot,resources,conflict);
                resources.remove(activity.resources.getResource(number));
            }
        } else {
            ActivityGroup conflictActivities = (ActivityGroup) conflict.clone();
            for (int i=0;i<resources.size();i++) 
                if (resources.isGroup(i)) 
                    for (int k=0;k<resources.getResources(i).size();k++)
                        conflictActivities.addIfIsNot(resources.getResources(i).get(k).getActivitiesAt(slot,activity.length));
                else
                    conflictActivities.addIfIsNot(resources.getResource(i).getActivitiesAt(slot,activity.length));
            config.log("    GENER: resources="+resources.toString()+" (conflict: "+conflictActivities+")");

            long start=System.currentTimeMillis();
              valueSelection.addValue(slot,resources,problem.dependences,conflictActivities);
            selectValueTime+=(System.currentTimeMillis()-start);
        }
    }

    /** Rekuzivn metoda, kter generuje mon umstn aktivity a pidv je do heuristiky vbru umstn.
     * @param activity nenaplnovan aktivita, pro kterou se umstn hled
     * @param valueSelection heuristika vbru umstn
     * @param slot poten slot, pro kter se umstn generuj
     * @param resources ji vybran zdroje (pro rekurzi, volno s new ActivityResources())
     * @param conflict mnoina ji nalezench konfliktnch aktivit
     * @param useResource zdroj, kter mus bt pouit (pokud je na vbr)
     */    
    private void generate(Activity activity, ValueSelectionInterface valueSelection, int slot, ActivityResources resources, ActivityGroup conflict, Resource useResource) throws TimetableException {
        if (useResource==null) {
            generate(activity, valueSelection, slot, resources, conflict);
            return;
        }
        int number= resources.size();
        if (number<activity.resources.size()) {
            if (activity.resources.isGroup(number)) {
                if (activity.resources.getResources(number).conjunctive) {
                    resources.add(activity.resources.getResources(number));
                    generate(activity,valueSelection,slot,resources,conflict,useResource);
                    resources.remove(activity.resources.getResources(number));
                } else {
                    if (activity.resources.getResources(number).indexOf(useResource)>=0) {
                        //System.out.println("  Using resource "+useResource.shortCut);
                        resources.add(useResource);
                        generate(activity,valueSelection,slot,resources,conflict,useResource);
                        resources.remove(useResource);
                    } else {
                        for (int i=0;i<activity.resources.getResources(number).size();i++) 
                            if (activity.resources.getResources(number).get(i).canBeFreeAt(slot,activity.length)) {
                                resources.add(activity.resources.getResources(number).get(i));
                                generate(activity,valueSelection,slot,resources,conflict,useResource);
                                resources.remove(activity.resources.getResources(number).get(i));
                            }
                    }
                }
            } else {
                resources.add(activity.resources.getResource(number));
                generate(activity,valueSelection,slot,resources,conflict,useResource);
                resources.remove(activity.resources.getResource(number));
            }
        } else {
            ActivityGroup conflictActivities = (ActivityGroup) conflict.clone();
            for (int i=0;i<resources.size();i++) 
                if (resources.isGroup(i)) 
                    for (int k=0;k<resources.getResources(i).size();k++)
                        conflictActivities.addIfIsNot(resources.getResources(i).get(k).getActivitiesAt(slot,activity.length));
                else
                    conflictActivities.addIfIsNot(resources.getResource(i).getActivitiesAt(slot,activity.length));
            config.log("    GENER: resources="+resources.toString()+" (conflict: "+conflictActivities+")");

            long start=System.currentTimeMillis();
              valueSelection.addValue(slot,resources,problem.dependences,conflictActivities);
            selectValueTime+=(System.currentTimeMillis()-start);
        }
    }
}