package ifs.solution;

import ifs.util.DataProperties;

/**
 * General implementation of solution comparator for minimal perturbation problem.
 * <br><br>
 * The solution is better than the best ever found solution when it has more variables assigned. In the case, when both
 * solutions have the same number of assigned variables, the one with smaller number of perturbations (i.e., variables
 * assigned to non-initial values) is selected. When all solution have the same number of assigned variables and 
 * number of perturbations, better solution is the one with smaller total value, i.e., the sum of {@link ifs.model.Value#toInt()} 
 * over all assigned variables.
 *
 * @see Solution
 * @see ifs.solver.Solver
 *
 * @author <a href="mailto:muller@ktiml.mff.cuni.cz">Tomas Muller</a>
 * @version 1.0
 */
public class MPPSolutionComparator implements SolutionComparator {
    
    public MPPSolutionComparator() {}
    /** No parameters are used so far. */
    public MPPSolutionComparator(DataProperties properties) {
    }
    
    public boolean isBetterThanBestSolution(Solution currentSolution) {
        if (currentSolution.getBestInfo()==null) return true;
        int unassigned = currentSolution.getModel().unassignedVariables().size();
        if (currentSolution.getModel().getBestUnassignedVariables()!=unassigned)
            return currentSolution.getModel().getBestUnassignedVariables()>unassigned;
        int pert = currentSolution.getModel().perturbVariables().size();
        if (currentSolution.getModel().getBestPerturbations()!=pert)
            return currentSolution.getModel().getBestPerturbations()>pert;
        return currentSolution.getModel().getTotalValue()<currentSolution.getBestValue();
    }
    
}
